#include "Polinom.h"

namespace algebra{
	
	/* Staticki clanovi klase se moraju definisati izvan klase, kako bi se za njih
	 u statickoj zoni memorije rezervisao prostor. Tom prilikom se clanu moze
	 pridruziti i inicijalna vrednost (ako se ne navede, podrazumeva se 0 za 
	 primitivne tipove, a za klasne tipove se poziva podrazumevani konstruktor).
	 */
double Polinom::_nula = 0.0;
	/* treci argument je opcioni i koristimo ga samo kod operatora [] kada treba povecati
	 polinom na veci stepen, novon je onda taj novi stepen, ako je 0, onda je stepen polinoma
	 vrednost prvog argumenta (podrazumevanu vrednost opcionog argumenta postavljamo pri deklaraciji
	 i ne ponavljamo pri definiciji funkcije)
	 */
void Polinom::init(int n, const double *koef){
	_n = n;
	/* alociramo memoriju za niz koeficijenata */
	_koef = new double[n+1];
	
	/* u slucaju poziva konstruktora Polinom(int), posto se ne prosledjuje niz 
	 koeficijenata, potrebno je sve postaviti na 0 jer new ne radi inicijalizaciju
	 */
	if(koef==NULL)
		for(int i = 0; i <= n; i++)
			_koef[i] = 0;
	else
		/* kada je prosledjen, postavljamo koeficijente 
		 kopiranjem clan po clan iz niza koef */
		for(int i = 0; i <= n; i++)
			_koef[i] = koef[i];

}

void Polinom::deinit(){
	/* oslobadjamo pridruzen objekat, tj. niz koeficijenata */
	delete [] _koef;
}

Polinom::Polinom(int n, const double * koef){
	init(n, koef);
}

Polinom::Polinom(int n){
	init(n);

}

Polinom::Polinom(const Polinom & p){
	init(p._n, p._koef);
}

int Polinom::vratiStepen() const{
	return _n;
}

/* Operator indeksiranja se moze definisati za bilo koju klasu, 
 i njime se omogucava sintaksa oblika p[k], gde je p objekat klase                *
 (u nasem slucaju, Polinom), a k podatak ciji tip odgovara tipu 
 parametra operatora (u nasem slucaju, int). Mogu se definisati indeksni
 operatori ciji je parametar proizvoljnog tipa (npr, stringovi, i sl.).
 Operator indeksiranja se mora definisati kao clan klase (ne moze biti
 definisan van klase). 
 
 Nas operator indeksiranja vraca referencu na double. Time se izraz
 p[k] smatra referencom na odgovarajuci (k-ti) koeficient polinoma, tj.
 predstavlja alias za sam k-ti koeficient. To znaci da je dozvoljeno 
 napisati:
 
 p[k] = 3;
 
 tj. promeniti vrednost odgovarajuceg koeficienta polinoma. Ukoliko 
 je k vece od stepena polinoma, tj. ne postoji odgovarajuci koeficient,
 tada se dimenzija polinoma povecava na k i vraca referenca na k-ti koeficijent */
double & Polinom::operator [](int k){
	if(k > _n){
		/* sacuvamo stari polinom */
		Polinom pom(*this); 
		/* obrisemo stare koeficijente */
		deinit();
		/* pravimo polinom stepena k, init sve inicijalizuje na 0 */
		init(k);
		/* ostalo je da kopiramo stare koeficijente iz pom */
		for(int i = 0; i <= pom._n; i++)
			_koef[i] = pom._koef[i];
	}
	return _koef[k];
}

/* Kada imamo dve funkcije clanice (ili dva operatora) koje se isto zovu,
 i imaju istu** listu parametara, one se i dalje mogu razlikovati, ukoliko
 je jedna od njih deklarisana kao const, a druga nije. Ne zaboravimo da
 const deklaracija cini da je this pokazivac na konstantan objekat klase,
 dok je u suprotnom this pokazivac na nekonstantan objekat klase. S 
 obzirom da je this pokazivac takodje (implicitni) parametar funkcije
 (operatora), po njegovom tipu se takodje dve funkcije (ili operatora)
 mogu razlikovati. Rezultujuca semantika je sledeca: ako se poziva funkcija
 za konstantan objekat, tada ce se pozivati ona varijanta koja je 
 deklarisana sa const (jer se nekonstantna i ne moze pozvati, to bi 
 zahtevalo konverziju implicitnog pokazivaca this iz pokazivaca na 
 konstantan u pokazivac na nekonstantan objekat). Ako se poziva funkcija
 za nekonstantna objekat, tada ce se pozivati ona varijanta koja nema
 const u svojoj deklaraciji (jer ona bolje odgovara pozivu, za drugu bi
 morala da se vrsi konverzija iz nekonstantnog u konstantan pokazivac).
 
 U nasem slucaju, prvi (nekonstantan) operator ne bi mogao da se pozove
 za konstantne objekte klase Polinom. Ako bismo samo njemu dodali const,
 tada bismo omogucili da se on poziva za svaki objekat (cak i za 
 konstantne), ali bi to onda bilo besmisleno, jer operator vraca 
 referencu na nekonstantan koeficient, koji se pomocu te reference moze
 promeniti. Ovo moze dovesti do semantickih gresaka koje je veoma tesko
 naci, jer konstantan operator ipak dozvoljava promenu objekta klase
 (zapravo, ne samog objekta, vec koeficienta koji je u nizu koji je 
 pridruzen objektu, ali nije deo samog objekta klase, pa zato i nije 
 automatski konstantan). Resenje je definisati jos jedan operator [], 
 deklarisati ga kao const, i promeniti njegov tip u const double &. 
 Ovaj operator ce se pozivati za konstantne objekte klase Polinom, 
 i vracace referencu na konstantan double, pa ga samim tim necemo moci
 menjati preko te reference (tj. obezbedjena je nepromenljivost 
 konstantnog objekta u sirem smislu, zajedno sa njegovim pridruzenim
 delovima).*/
const double & Polinom::operator [](int k) const{
	return k <= _n ? _koef[k] : _nula;
}

/*   Operator dodele kopiranjem je operator dodele koji omogucava da se jedan
 o bjekat kla*snog tipa dodeli drugom objektu iste klase. Ovaj operator
 uvek postoji, a poziva se prilikom svake dodele jednog objekta drugom.
 Ukoliko operator nije eksplicitno definisan, tada se poziva implicitno
 definisani operator dodele kopiranjem cija je semantika nasledjena od
 C-ovih struktura -- svaki nestaticki clan se prosto dodeli odgovarajucem
 nestatickom clanu objekta sa leve strane operatora dodele. Prilikom 
 dodele clan-po-clan, za svaki nestaticki clan klasnog tipa poziva se 
 operator dodele (implicitni ili eksplicitni) te klase. 
 
 Ukoliko je podrazumevana semantika neodgovarajuca, tada programer moze
 eksplicitno definisati svoj operator dodele kopirajem. U pitanju je
 operator = koji kao parametar ima objekat iste klase, ili referencu na
 isti (sto je tipicno). Po konvenciji, operator = bi trebalo da vraca
 referencu na objekat klase za koji je i pozvan (kako bismo mogli da
 ga koristimo u obliku a = b = c, ili x + (a = b), i sl. kao sto smo 
 navikli u C-u. 
 
 Pored operatora dodele kopiranjem, moguce je definisati i druge operatore
 dodele (ciji je parametar nekog drugog tipa). Ovi operatori su prica za
 sebe i nemaju nikakve veze sa operatorom dodele kopiranjem. 
 
 Operatori dodele se moraju definisati unutar klase, tj. kao funkcije
 clanice. */
Polinom & Polinom::operator = (const Polinom & p){
	// Obicno se na pocetku proveri da nije u pitanju dodela oblika a = a,
	// koja ne bi trebalo da radi nista.
	if(&p != this){
		// Brisemo postojece koeficiente. 
		deinit();
		// Kreiramo novi niz koeficienata, a zatim, kao u konstruktoru kopiranja,
		// kopiramo koeficiente. Upravo je ovo razlog zasto nam je potrebna
		// eksplicinta definicija operatora dodele -- bez nje bi podrazumevani
		// operator dodele samo "prevezao" pokazivac na niz koeficienata polinoma
		// p, koji bi od tog trenutka bio zajednicki za oba polinoma ("plitko"
		// kopiranje). Takodje, stari niz koeficienata bi ostao negde u memoriji,
		// nedealociran.
		init(p._n, p._koef);
	}
	// Uvek vracamo *this, tj. upravo objekat klase za koji je operator 
	// pozvan. Ovim se referenca koja se vraca inicijalizuje upravo na ovaj
	// objekat.
	return *this;
}

// Operator sabiranja dva polinoma.
Polinom Polinom::operator + (const Polinom & p) const{
	/* stepen zbira je jednak vecem od stepena polinoma sabiraka */
	int stepen = vratiStepen() > p.vratiStepen() ? vratiStepen() : p.vratiStepen();
	Polinom rez(stepen);
  /* koeficijenti zbira su jednaki zbiru odgovarajucih koeficijenata sabiraka */  
	for(int i = 0; i <= stepen; i++)
		rez[i] = (*this)[i] + p[i];
    
	return rez;
}

/* Operator poziva funkcije () se takodje moze definisati za svaku klasu.
 O vo omogucava sintaksu oblika p(x), gde je p ob*j*ekat klase (u nasem
 slucaju Polinom), a x je podatak tipa parametra operatora. Moguce 
 je definisati ovaj operator i sa vise parametara, tako da se poziva sa
 npr. p(x, y, z). Drugim recima, ovim operatorom se omogucava da se objekat
 klase sintaksno ponasa kao da je funkcija.
 
 U nasem slucaju, operator () se koristi za racunanje vrednosti polinoma
 u tacki. Parametar je jedan broj tipa double. */
double Polinom::operator () (double x) const{
	double v = 0;
	// Racunamo vrednost polinoma (Hornerova shema)
	for(int i = vratiStepen(); i >= 0; i--)
		v = v * x + (*this)[i];
	return v;
}

/* Destrukcija je suprotna operacija od konstrukcije, ona se izvrsava svaki
 p ut neposredno *pre dealokacije objekta (u slucaju automatskih objekata,
 neposredno pre zavrsetka funkcije ili bloka u kojoj je taj podatak
 definisan, u slucaju dinamickih objekata, prilikom pozivanja operatora
 delete. Ukoliko programer zeli da se nesto izvsi u toku destrukcije
 (pre destrukcije nestatickih podataka clanova i dealokacije samog 
 objekata), tada treba da definise specijalnu funkciju clanicu koja se
 zove destruktor. Njegovo ime se sastoji iz znaka ~ za kojim sledi ime
 klase. Destruktor nema parametre i ne moze se preklapati (dakle, moze 
 biti samo jedan). U nasem slucaju, destruktor omogucava dealokaciju
 dinamicke memorije (niza koeficienata). Ova memorija nije deo samog 
 objekta, pa se nece automatski obrisati kada se iz memorije brise sam
 objekat. Ako zelimo da se to desi, moramo to da uradimo rucno, u 
 destruktoru. */
Polinom::~Polinom(){
	deinit();
}

/* Operator ispisa polinoma */
std::ostream& operator << (std::ostream& ostr, const Polinom& p){
	for(int i = p.vratiStepen(); i >= 0; i--){
		ostr << p[i];
		if(i != 0){
			ostr << "*x^" << i << "+ ";
		}
		
	}
	return ostr;
}

}
