// Jednostavan primer koji pokusava da izracuna dvostruku sumu i proizvod niza
// Demonstrira kako izgleda uopsteni sablon koji koriscenjem ogranicenog
// broja radnika pokusava konkurentno da odradi posao

package main

import (
	"fmt"
	"runtime"
	"sync"
)

// Paket nam treba kao struktura jer predstavlja jedinicu posla
// Ideja je da u paket stavimo sve podatke koji su potrebni da bi se
// izvrsilo jedno izracunavanje
// Takodje se stavlja polje koje simbolise "element posle posednjeg"
// polje valid ako je false znaci da nije u pitanju pravi podatak nego
// signaliziramo da nema vise posla
type paket struct {
	info  int
	valid bool
}

// WaitGroup ne moramo da pravimo poseban za sabiranje i mnozenje jer
// na kraju svakako isti program ceka sve da se zavrsi na istom mestu,
// mada ne bi bilo pogresno i napraviti odvojene

var wg sync.WaitGroup

var suma int

// Mutexe pravimo odvojene, jer ne zelimo da isto usko grlo
// bude i u kriticnoj sekciji za sabiranje i u sekciji za mnozenje
// operacije su potpuno nepovezane i ne operisu nad istim podacima
// kada bi delili isti mutex, medjusobno bi morali da se cekaju
var mutexSaberi sync.Mutex

// Funkcija predstavlja jednu nit
func saberi(c chan paket) {
	// U beskonacnoj petlji dohvatamo jedan po jedan element
	for {
		p := <-c
		// Ukoliko naidjemo na paket koji nije validan
		// to znaci da smo zavrsili obradu i da nema vise
		// podataka, nit moze da zavrsi svoj posao
		// Obavezno moramo da vratimo taj nevalidni paket
		// nazad u channel, jer je citanje potrosno, a treba
		// i ostale niti da vide da je doslo do kraja
		if !p.valid {
			c <- paket{0, false}
			break
		} else {
			broj := 2 * p.info

			mutexSaberi.Lock()
			suma += broj
			mutexSaberi.Unlock()
		}
	}

	wg.Done()
}

var proizvod int = 1

// Poseban mutex za mnozenje
var mutexPomnozi sync.Mutex

func pomnozi(c chan paket) {
	for {
		p := <-c
		if !p.valid {
			c <- paket{0, false}
			break
		} else {
			broj := p.info

			mutexSaberi.Lock()
			proizvod *= broj
			mutexSaberi.Unlock()
		}
	}

	wg.Done()
}

func main() {
	var numWorkers = runtime.NumCPU()

	// Ogranicavamo velicinu kanala na broj radnika
	// Nema potrebe da bude vise prostora nego sto
	// zapravo ima radnika da rade posao,
	// mada u ovom slucaju ne bi bila greska staviti malo vise prostora
	var cSaberi = make(chan paket, numWorkers)
	var cPomnozi = make(chan paket, numWorkers)
	wg.Add(2 * numWorkers)
	for i := 0; i != numWorkers; i++ {
		go saberi(cSaberi)
		go pomnozi(cPomnozi)
	}

	for {
		var broj int
		fmt.Scanf("%d", &broj)
		if broj == 0 {
			p := paket{0, false}
			cSaberi <- p
			cPomnozi <- p
			break
		} else {
			p := paket{broj, true}
			cSaberi <- p
			cPomnozi <- p
		}
	}

	wg.Wait()
	fmt.Printf("Suma: %v; Proizvod: %v\n", suma, proizvod)
}
