﻿//* *//* *//* *//* *//* *//* *//* *//* *//* *//* *//* *//
//                                                     //
//  Multithreading Application in Modern Applications  //
//                  Nikola Premcevski                  //
//                 February  2nd, 2016                 //
//                                                     //
//* *//* *//* *//* *//* *//* *//* *//* *//* *//* *//* *//

import System.Collections.Generic;
import System.IO;
import System.Threading;
import UnityEngine.UI;

// Main References
var OptionsPanelTransform : Transform;
var FinalTextureTransform : Transform;
var CanvasTransform : Transform;
var LabelSliceResolution : Text;
var LabelHueOffset : Text;
var SliderHueOffset : Slider;
var ToggleRealtime : Toggle;
// Prefabs
var ActiveThreadSpritePrefab : Transform;
var Error404Texture : Texture2D;
// Settings
private var ImageURL : String;
private var TextureResolution : Vector2; // Used for the placeholder if no image is loaded
private var SliceResolution : int; // Affects the total amount of threads needed
private var ImageFilterMode : int;
private var HueOffset : float;
private var RealtimeMode : boolean;
private var DebugMode : boolean;
// Internal
private var OriginalTexture : Texture2D;
private var OriginalTexturePixels : Color32[];
private var FinalTexture : Texture2D;
private var FinalTexturePixels : Color32[];
private var ProcessorsCount : byte;
private var SlicesQueue : List.<Rect>;
private var Threads : List.<Thread>;
private var ThreadCallbacks : List.<NewThreadData>;
private var ActiveThreadsTransforms : Transform[];
var TimeTaken : float;
var Pause : boolean;
private var TargetSize : float;
private var HueOffsetArray : float[];

class NewThreadData{
	var ThreadID : byte; // Limits the number of usable cores to 256
	var SliceRect : Rect;

	function NewThreadData(t : byte, r : Rect){
		ThreadID = t;
		SliceRect = r;
	}
}

function Start(){
	var i : int;
	// Settings
	ImageURL = "http://dzoni94.redirectme.net/MAiMA.jpg";
	TextureResolution = Vector2.zero;
	SliceResolution = 30;
	ImageFilterMode = 0;
	HueOffset = 0;
	RealtimeMode = false;
	DebugMode = true;
	TargetSize = 1;
	// Internal
	ProcessorsCount = SystemInfo.processorCount;
	ActiveThreadsTransforms = new Transform[ProcessorsCount];
	SlicesQueue = new List.<Rect>();
	Threads = new List.<Thread>();
	ThreadCallbacks = new List.<NewThreadData>();
	for(i=0; i<ProcessorsCount; i++){
		var go : Transform = Instantiate(ActiveThreadSpritePrefab);
		go.name = String.Format("AT{0}", i);
		ActiveThreadsTransforms[i] = go;
		go.SetParent(FinalTextureTransform);
		go.gameObject.SetActive(false);
	}
}
function Update(){
	if(Input.GetKeyDown(KeyCode.O)) OptionsPanelTransform.gameObject.SetActive(!OptionsPanelTransform.gameObject.activeSelf);
	if(Input.GetKeyDown(KeyCode.S)) SaveImage();
	if(Input.GetAxis("Mouse ScrollWheel")>0) TargetSize *= 1.1;
	else if(Input.GetAxis("Mouse ScrollWheel")<0) TargetSize *= 0.9;
	TargetSize = Mathf.Clamp(TargetSize, 0.25, 10);
	FinalTextureTransform.localScale = Vector3.Lerp(FinalTextureTransform.localScale, Vector3(TargetSize, TargetSize, 1), 5*Time.deltaTime);
	if(ThreadCallbacks.Count > 0) ParseThreadCallback();
	if(!Pause) TimeTaken += Time.deltaTime;
}
function LoadImage(){
	var i : int;
	// Remove the old image and load a new one, or create a placeholder
	FinalTextureTransform.gameObject.SetActive(false);
	Destroy(FinalTexture);
	if(!String.IsNullOrEmpty(ImageURL)){
		var www : WWW = new WWW(ImageURL);
		yield www;
		if(String.IsNullOrEmpty(www.error)){
			OriginalTexture = www.texture;
			FinalTexture = www.texture;
			TextureResolution = Vector2(FinalTexture.width, FinalTexture.height);
		}
		else LoadPlaceholderImage();
	}
	else LoadPlaceholderImage();
	FinalTexture.wrapMode = TextureWrapMode.Clamp;
	FinalTexture.filterMode = FilterMode.Point;
	OriginalTexturePixels = OriginalTexture.GetPixels32();
	FinalTexturePixels = FinalTexture.GetPixels32();
	FinalTextureTransform.GetComponent(RawImage).texture = FinalTexture;
	FinalTextureTransform.sizeDelta = TextureResolution;
	FinalTextureTransform.gameObject.SetActive(true);
	// Place the thread sprites over the image
	for(i=0; i<ProcessorsCount; i++) ActiveThreadsTransforms[i].SetAsLastSibling();
}
function LoadPlaceholderImage(){
	OriginalTexture = new Texture2D(4, 4, TextureFormat.RGBA32, false);
	OriginalTexture.LoadImage(Error404Texture.EncodeToPNG());
	FinalTexture = new Texture2D(4, 4, TextureFormat.RGBA32, false);
	FinalTexture.LoadImage(Error404Texture.EncodeToPNG());
	TextureResolution = Vector2(OriginalTexture.width, OriginalTexture.height);
}
function FilterImage(){
	if(!RealtimeMode){
		if(ImageFilterMode == 5) CreateHueArray(HueOffset);
	}
	TimeTaken = 0;
	Pause = false;
	var i : int;
	var j : int;
	for(i=0; i<ProcessorsCount; i++) ActiveThreadsTransforms[i].sizeDelta = Vector2(SliceResolution, SliceResolution);
	var slicesCount : Vector2 = Vector2(Mathf.Ceil(TextureResolution.x/SliceResolution), Mathf.Ceil(TextureResolution.y/SliceResolution));
	for(i=0; i<slicesCount.y; i++){
		for(j=0; j<slicesCount.x; j++){
			var width : int = (j+1)*SliceResolution<=TextureResolution.x?SliceResolution:TextureResolution.x-j*SliceResolution;
			var height : int = (i+1)*SliceResolution<=TextureResolution.y?SliceResolution:TextureResolution.y-i*SliceResolution;
			SlicesQueue.Add(Rect(j*SliceResolution, i*SliceResolution, width, height));
		}
	}
	var min : int = Mathf.Min(slicesCount.x*slicesCount.y, ProcessorsCount);
	for(i=0; i<min; i++) StartThread(i);
}
// Thread functions
function StartThread(threadID : byte){
	var sliceRect = SlicesQueue[0];
	SlicesQueue.RemoveAt(0);
	var data : NewThreadData = new NewThreadData(threadID, sliceRect);
	var thread : Thread = new Thread(ThreadFunction);
	Threads.Add(thread);
	thread.Start(data);
	if(DebugMode){
		var posX : int = sliceRect.x - TextureResolution.x/2;
		var posY : int = sliceRect.y - TextureResolution.y/2;
		ActiveThreadsTransforms[threadID].localPosition = Vector3(posX, posY - 0.5, 0);
		ActiveThreadsTransforms[threadID].sizeDelta = Vector2(sliceRect.width, sliceRect.height);
		ActiveThreadsTransforms[threadID].gameObject.SetActive(true);
	}
}
function ThreadFunction(data : NewThreadData){
	var i : int;
	var j : int;
	var limitY : int = data.SliceRect.y + data.SliceRect.height;
	var limitX : int = data.SliceRect.x + data.SliceRect.width;
	for(i=data.SliceRect.y; i<limitY; i++){
		for(j=data.SliceRect.x; j<limitX; j++){
			var index : int = i*TextureResolution.x + j;
			if(ImageFilterMode == 1) FinalTexturePixels[index] = InvertPixel32(FinalTexturePixels[index], false);
			else if(ImageFilterMode == 2) FinalTexturePixels[index] = InvertPixel32(FinalTexturePixels[index], true);
			else if(ImageFilterMode == 3) FinalTexturePixels[index] = SquarePixel32(FinalTexturePixels[index], false);
			else if(ImageFilterMode == 4) FinalTexturePixels[index] = SquarePixel32(FinalTexturePixels[index], true);
			else if(ImageFilterMode == 5) FinalTexturePixels[index] = HuePixel32(OriginalTexturePixels[index], HueOffset);
		}
	}
	ThreadCallbacks.Add(data);
}
function ParseThreadCallback(){
	var data : NewThreadData = ThreadCallbacks[0];
	FinalTexture.SetPixels32(FinalTexturePixels);
	FinalTexture.Apply();
	ThreadCallbacks.RemoveAt(0);
	Threads.RemoveAt(0);
	ActiveThreadsTransforms[data.ThreadID].gameObject.SetActive(false);
	if(SlicesQueue.Count > 0) StartThread(data.ThreadID);
	else Pause = true;
}
// Pixel functions
function InvertPixel32(pixel : Color32) : Color32{ return Color32(255 - pixel.r, 255 - pixel.g, 255 - pixel.b, 255); }
function SquarePixel32(pixel : Color32) : Color32{
	var r : float = pixel.r/255.0;
	var g : float = pixel.g/255.0;
	var b : float = pixel.b/255.0;
	return Color32(r*r*255, g*g*255, b*b*255, 255);
}
function InvertPixel32(pixel : Color32, alpha : boolean) : Color32{ return Color32(255-pixel.r, 255-pixel.g, 255-pixel.b, alpha?255-pixel.a:255); }
function SquarePixel32(pixel : Color32, alpha : boolean) : Color32{
	var r : float = pixel.r/255.0;
	var g : float = pixel.g/255.0;
	var b : float = pixel.b/255.0;
	var a : float = pixel.a/255.0;
	return Color32(r*r*255, g*g*255, b*b*255, alpha?a*a*255:255);
}
function HuePixel32(pixel : Color32, offset : int) : Color32{
	var color : Color32;
	color.r = Mathf.Clamp(pixel.r*HueOffsetArray[0] + pixel.g*HueOffsetArray[1] + pixel.b*HueOffsetArray[2], 0, 255);
	color.g = Mathf.Clamp(pixel.r*HueOffsetArray[2] + pixel.g*HueOffsetArray[0] + pixel.b*HueOffsetArray[1], 0, 255);
	color.b = Mathf.Clamp(pixel.r*HueOffsetArray[1] + pixel.g*HueOffsetArray[2] + pixel.b*HueOffsetArray[0], 0, 255);
	return color;
}
function CreateHueArray(offset : int){
	var cosA : float = Mathf.Cos(offset*Mathf.Deg2Rad);
	var sinA : float = Mathf.Sin(offset*Mathf.Deg2Rad);
	var third : float = 0.33333333333333333333333333333333;
	var rootThird : float = 0.57735026918962576450914878050196;
	var x : float = (1-cosA)*third;
	var y : float = rootThird*sinA;
	HueOffsetArray = new float[3];
	HueOffsetArray[0] = x+cosA;
	HueOffsetArray[1] = x-y;
	HueOffsetArray[2] = x+y;
}
// UI Wrapper functions
function WrapperSetImageURL(text : String){ ImageURL = text; }
function WrapperLoadImage(){ LoadImage(); }
function WrapperFilterMode(value : int){
	ImageFilterMode = value;
	SliderHueOffset.gameObject.SetActive(value==5);
	LabelHueOffset.gameObject.SetActive(value==5);
	ToggleRealtime.gameObject.SetActive(value==5);
}
function WrapperSliceResolution(value : float){ SliceResolution = 30*value; LabelSliceResolution.text = String.Format("Slice Resolution: {0}px", parseInt(30*value)); }
function WrapperHueOffset(value : float){
	HueOffset = value;
	LabelHueOffset.text = String.Format("Hue Offset: {0}deg", parseInt(value));
	if(RealtimeMode){
		CreateHueArray(HueOffset);
		FilterImage();
	}
}
function WrapperRealtimeMode(value : boolean){ RealtimeMode = value; }
function WrapperDebugMode(value : boolean){ DebugMode = value; }
function WrapperFilterImage(){ FilterImage(); }
function WrapperStop(){ SlicesQueue.Clear(); for(var i : int=0; i<Threads.Count; i++) Threads[i].Abort(); }
function WrapperSave(){ SaveImage(); }
// Miscc
function SaveImage(){
	var path : String = String.Format("{0}/image.png", Application.dataPath);
	var data : byte[] = FinalTexture.EncodeToPNG();
	if(data) File.WriteAllBytes(path, data);
}