-- uvodimo novi tip podataka koji predstavlja mapu realizovanu u obliku sortiranog binarnog stabla
-- svaki cvor sadrzi kljuc k i vrednost njemu pridruzenu v
-- kao i u mapi, nema duplikata kljuceva, tj. cvorova sa istom vrednoscu k u stablu 

-- primer objekata tipa Map:
-- Null - prazna mapa
-- (Node 2 3 Null Null) - mapa koja sadrzi kljuc 2 sa vrednoscu 3
-- (Node 2 3 (Node 1 5 Null Null) (Node 7 8 Null Null)) - mapa koja sadrzi kljuceve 1, 2, 3 sa vrednostima 5, 3, 8 redom

data Map k v = Null 
	| Node k v (Map k v) (Map k v)
	-- dodajemo nas tip podataka u tipski razred Show
	-- sto omogucava prikaz objekata tipa Map u terminalu
	deriving Show

null Null = True
null _ = False

-- primer funkcije za transformaciju koja koristi kljuc, tekucu i novu vrednost
f k v v' = k + v - v'

-- da bi stablo bilo balansirano
-- cvorove sa manjim kljucem od kljuca u korenu dodajemo u levo podstablo
-- cvorove sa vecim kljucem od kljuca u korenu dodajemo u desno podstablo
-- ako naidjemo na kljuc koji treba dodati, samo menjamo vrednost na osnovu funkcije f
insertWithKey f k v Null = Node k v Null Null
insertWithKey f k v (Node k' v' l r)
	| k < k'  = Node k' v' (insertWithKey f k v l) r
	| k > k'  = Node k' v' l (insertWithKey f k v r)
	| otherwise = Node k' (f k' v' v) l r

-- funkcija za transformaciju ne koristi vrednost kljuca
insertWith :: Ord k => (v -> v -> v) -> k -> v -> Map k v -> Map k v
insertWith f = insertWithKey (\ k v v' -> f v v')

-- za nepostojeci kljuc dodajemo novi cvor sa kljucem k i vrednoscu v
-- ne koristi se funkcija za transformaciju 
insert :: Ord k => k -> v -> Map k v -> Map k v
insert = insertWith (\ v v' -> v')

-- na osnovu parova iz liste pravimo cvorove u stablu
-- foldl koristi tekuci element liste (jedan par kljuc, vrednost)
-- i do tada napravljeno stablo na osnovu prethodnih elemenata iz liste  
fromList :: Ord k => [(k, v)] -> Map k v
fromList = foldl (\ m (k,v)  -> insert k v m) Null


-- Maybe tip podataka koristimo kad je potrebno naglasiti 
-- da za neke argumente ne postoji vrednost funkcije
-- data Maybe a = Nothing | Just a
search :: Ord k => Map k v -> k -> Maybe v
search Null _ = Nothing
search (Node k v l r) k' 
	| k < k' = search r k'
	| k > k' = search l k'
	| otherwise = Just v

-- dodefinisali smo funkciju search da u slucaju Nothing vraca parametar d
findWithDefault :: Ord k => v -> k -> Map k v -> v
findWithDefault d k m = case search m k of 
	Nothing -> d 
	Just v -> v

-- ukoliko je search vratio Nothing - nije pronasao kljuc k pa zbog toga ne postoji ni vrednost njemu pridruzena
-- inace, pronasao je trazeni kljuc
member :: Ord k => Map k v -> (k -> Bool)
member m k = case search m k of 
	Nothing -> False
	Just _ -> True

adjustWithKey :: Ord k => (k -> v -> v) -> k -> Map k v -> Map k v
adjustWithKey f k Null = Null
adjustWithKey f k (Node k' v' l r)
   | k < k'   = Node k' v' (adjustWithKey f k l) r
   | k > k'   = Node k' v' l (adjustWithKey f k r)
   | k == k'  = Node k' (f k v') l r

adjustAll :: (k -> v -> v) -> Map k v -> Map k v
adjustAll f Null = Null
adjustAll f (Node k v l r) = Node k (f k v) (adjustAll f l) (adjustAll f r)

-- najmanji kljuc se nalazi u najlevljem cvoru stabla
deleteMin (Node k v Null r) = ( (k, v), r)
deleteMin (Node k v l r) = ( (k', v'), Node k v o r)
	where ((k', v'), o) = deleteMin l

-- u slucaju da se brise cvor koji ima levo ili desno podstablo
-- ideja: da se pronadje minimum desnog podstabla
-- i da se on postavi za novi koreni cvor jer se time cuva sortiranost kljuceva
delete k Null = Null
delete k (Node k' v' l r)
	| k < k'  = Node k' v' (delete k l) r
	| k > k'  = Node k' v' l (delete k r)
	| otherwise =  case r of 
			Null -> l
			_ -> Node  k'' v'' l d
				where ((k'', v''), d) = deleteMin r

-- agregacija
foldMap :: (a -> k -> v -> a) -> a -> Map k v -> a
foldMap _ i Null = i
foldMap f i (Node k v l r) = foldMap f (f (foldMap f i l) k v) r

-- velicina mape - broj kljuceva
size :: Map k v -> Int
size = foldMap (\ a k v -> a + 1) 0

-- pravljenje liste parova od mape
toList :: Ord k => Map k v -> [(k, v)]
toList = reverse . foldMap (\ a k v -> (k, v) : a) []

-- unija dve mape - unija dva stabla pri cemu se odrzava sortiranost
union :: Ord k => Map k v -> Map k v -> Map k v
union m1 m2 = foldMap (\ a k v -> insert k v a) m1 m2


